/* --------------------------------------------------------------
  Modal.js 2019-06-07
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2019 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/


'use strict';

export default {
	/**
	 * Show a modal with the contents of a template.
	 *
	 * @param {jQuery} $template The template element to be used for the modal.
	 * @param {Object} data Contains the template data.
	 *
	 * @return {jQuery} Returns the selector of the
	 */
	show($template, data) {
		StyleEdit.Validator.isObject($template);
		StyleEdit.Validator.isObject(data);
		
		const html = Mustache.render($template.html(), data);
		
		const $modal = $(`<div class="modal">${html}</div>`);
		
		$modal.modal({backdrop: 'export function'});
		
		return $modal;
	},
	
	/**
	 * Hide a displayed modal with the provided selector.
	 *
	 * @param {jQuery} $modal
	 */
	hide($modal) {
		StyleEdit.Validator.isObject($modal);
		
		$modal.on('hidden.bs.modal', () => $modal.remove());
		
		$modal.modal('hide');
	},
	
	/**
	 * Display the prompt template as a modal to the user.
	 *
	 * The user will have to confirm the modal in order to proceed, he can also press
	 * the "Cancel" button.
	 *
	 * @param {String} title The title of the modal.
	 * @param {String} message The message of the modal.
	 *
	 * @return {jQuery.Promise} Returns a promise object that will be resolved if the user confirms the prompt.
	 */
	prompt(title, message) {
		StyleEdit.Validator.isString(title);
		StyleEdit.Validator.isString(message);
		
		const deferred = $.Deferred();
		const template = $('#prompt-modal-template').html();
		const data = {
			title,
			message,
			option_yes: StyleEdit.Language.translate('option_yes', 'style_edit'),
			option_no: StyleEdit.Language.translate('option_no', 'style_edit')
		};
		const html = Mustache.render(template, data);
		const $modal = $(html);
		
		$modal.modal({backdrop: 'export function'});
		
		$modal.find('.btn.yes')
			.off('click')
			.on('click', () => {
				$modal.modal('hide').remove();
				deferred.resolve();
			});
		
		$modal.find('.btn.no')
			.off('click')
			.on('click', () => {
				$modal.modal('hide').remove();
				deferred.reject();
			});
		
		return deferred.promise();
	},
	
	/**
	 * Display a simple message modal to the user.
	 *
	 * The user will only be able to click on an "OK" button to dismiss the modal.
	 *
	 * @param {String} title The title of the modal.
	 * @param {String} message The message of the modal.
	 *
	 * @return {jQuery.Promise} Returns a promise object that will be resolved once the user clicks on the "OK" button.
	 */
	message(title, message) {
		StyleEdit.Validator.isString(title);
		StyleEdit.Validator.isString(message);
		
		const deferred = $.Deferred();
		const template = $('#message-modal-template').html();
		const data = {
			title,
			message,
			option_ok: StyleEdit.Language.translate('option_ok', 'style_edit')
		};
		const html = Mustache.render(template, data);
		const $modal = $(`<div class="modal">${html}</div>`);
		
		$modal.modal({backdrop: 'export function'});
		
		$modal.find('.btn.ok')
			.off('click')
			.on('click', () => {
				$modal.modal('hide').remove();
				deferred.resolve();
			})
			.focus();
		
		return deferred.promise();
	}	
}
